#-----------------------------------------------------------------------------
#
#   File:           student.py
#   Module:         student
#   Package:        N/A
#   Project:        UC-05-2018 UR5 Robot Coffee Cart
#   Workspace:      Shotbot.rdk
#
#   Software:       Polyscope 3.0.16040
#   Firmware:       URControl 3.0.16040
#   Hardware:       UR5 (CB3UR5)
#
#   Target:         N/A
#   Environment:    RoboDK 3.4.7
#                   Python 3.4
#                   Doxygen 1.8.14
#
#   Author:         Rodney Elliott
#   Organisation:   University of Canterbury
#   Date:           23 October 2018
#
#-----------------------------------------------------------------------------
#
#   Copyright:      Rodney Elliott 2018
#                   University of Canterbury
#
#   This file is part of Shotbot.
#
#   Shotbot is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Shotbot is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Shotbot. If not, see <https://www.gnu.org/licenses/>.
#
#-----------------------------------------------------------------------------
##
#   @mainpage Coffeebot
#   @section sec_introduction Introduction
#
#   The goal of the 2019 UR5 Laboratory Assignment is to use the UR5 Robot
#   Coffee Cart to pull a double shot espresso whose volume should not be
#   greater than 25ml.
#
#   In order to accomplish this goal, students will need to use the RoboDK
#   station generated by the 'student' Python module and the various other
#   assignment resources to:
#
#   - Compute the homogenous transform (HT) for each and every operational
#     feature within the robot workspace relative to the robot base frame.
#   - Program the robot to perform the numerous actions needed to produce
#     a short black, using the computed HTs to guide its motion.
#
#   @section sec_details Details
#
#   TODO: Add further details.
#

##
#   @package student
#   @brief The student Python module.
#
#   <H3> Introduction </H3>
#
#   A module written for use by students enrolled in the ENMT482 Robotics
#   course offered by the University of Canterbury. Its purpose is to generate
#   a RoboDK station tree containing the tools, frames, geometry, targets, and
#   programs needed to complete the UR5 Laboratory Assignment.
#
#   <H3> Use </H3>
#
#   To generate the station tree, simply double-click on the Python file from
#   within RoboDK, or alternatively right-click on the file and select 'Run
#   Python script' from the pop-up menu.
#

#-----------------------------------------------------------------------------
#   Modules
#-----------------------------------------------------------------------------
##
#   @brief RoboDK interface.
#
from robolink import *

##
#   @brief Robotics toolbox.
#
from robodk import *

##
#   @brief Useful math functions.
#
from math import radians

#-----------------------------------------------------------------------------
#   Globals
#-----------------------------------------------------------------------------
PATH = 'C:/RoboDK/User/'

ATTACH_TOOL = 4
DETACH_TOOL = 5
EXTEND_RAM = 6
RETRACT_RAM = 7

#-----------------------------------------------------------------------------
#   Functions
#-----------------------------------------------------------------------------
##
#   @brief Add UR5 robot 2014350238 to the station.
#
#   Adds UR5 robot serial no. 2014350238 to the currently-active station. The
#   '.robot' file contains the Denavit-Hartenburg (DH) parameters specific to
#   this particular robot arm. These DH parameters differ to those found in
#   the generic RoboDK UR5 '.robot' file.
#
def add_ur5_robot():
    station = RDK.ActiveStation()
    station.AddFile(PATH + 'UR5 2014350238.robot')
    
    ur5_2014350238_base = RDK.Item('UR5 2014350238 Base', ITEM_TYPE_FRAME)
    ur5_2014350238_base.setPose(eye(4))
    ur5_2014350238_base.setVisible(False)
    
    return

##
#   @brief Add master tool to UR5 robot 2014352038.
#
#   Adds a master tool to UR5 robot serial no. 2014352038. The master tool is
#   permanently affixed to the robot tool flange, and as such the master tool
#   geometry does not require a z-axis offset (c.f. the slave tools).
#
def add_ur5_master_tool():
    ur5_2014350238 = RDK.Item('UR5 2014350238', ITEM_TYPE_ROBOT)
    
    ur5_2014350238.AddFile(PATH + 'Tool Files/Master Tool.tool')
    master_tool = RDK.Item('Master Tool', ITEM_TYPE_TOOL)
    master_tool.setVisible(True, False)
    
    return

##
#   @brief Add slave tools to UR5 robot 2014352038.
#
#   Adds slave tools to UR5 robot serial no. 2014352038. Slave tools have a
#   z-axis geometry offset of 48.94mm relative to that of the master tool to
#   ensure they appear in the correct location in RoboDK when mated. The TCP
#   of each slave tool is defined such that it is centred on, and colinear
#   with, the master tool mating face.
#
#   Note that it is left as an exercise for the student to determine the tool
#   transforms necessary to bring the slave tool TCPs into alignment with the
#   salient tool features as illustrated in the UR5 Laboratory Assignment
#   support material.
#
def add_ur5_slave_tools():
    ur5_2014350238 = RDK.Item('UR5 2014350238', ITEM_TYPE_ROBOT)
            
    ur5_2014350238.AddFile(PATH + 'Tool Files/Grinder Tool.tool')
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_TOOL)
    grinder_tool.setVisible(False)
        
    ur5_2014350238.AddFile(PATH + 'Tool Files/Portafilter Tool.tool')
    portafilter_tool = RDK.Item('Portafilter Tool', ITEM_TYPE_TOOL)
    portafilter_tool.setVisible(False)
    
    ur5_2014350238.AddFile(PATH + 'Tool Files/Cup Tool Closed.tool')
    cup_tool_closed = RDK.Item('Cup Tool Closed', ITEM_TYPE_TOOL)
    cup_tool_closed.setVisible(False)
    ur5_2014350238.AddFile(PATH + 'Tool Files/Cup Tool Open.tool')
    cup_tool_open = RDK.Item('Cup Tool Open', ITEM_TYPE_TOOL)
    cup_tool_open.setVisible(False)
    
    return

##
#   @brief Add a paper cup to UR5 robot 2014350238.
#
#   Adds a paper cup to UR5 robot serial no. 2014352038. The cup geometry has
#   been offset to ensure that it appears in the correct location relative to
#   the cup tool in RoboDK.
#
def add_ur5_paper_cup():
    ur5_2014350238 = RDK.Item('UR5 2014350238', ITEM_TYPE_ROBOT)
            
    ur5_2014350238.AddFile(PATH + 'Tool Files/Paper Cup.tool')
    paper_cup = RDK.Item('Paper Cup', ITEM_TYPE_TOOL)
    paper_cup.setVisible(False)
    
    return

##
#   @brief Set the pose of the master tool.
#
#   The master tool has its TCP defined such that it is centred on, and
#   colinear with, the slave tool mating face.
#
def set_master_tool_pose():
    tcp = transl(0, 0, 48.94)
    
    tool = RDK.Item('Master Tool', ITEM_TYPE_TOOL)
    tool.setPoseTool(tcp)
    
    return

##
#   @brief Add the table to the station.
#
#   Adds the fundamental 'Table Top' frame and 'Table Plate' geometry to the
#   station. Additional frames, geometry, and targets may then be added by
#   other functions.
#
def add_table():
    ur5_2014350238_base = RDK.Item('UR5 2014350238 Base', ITEM_TYPE_FRAME)
    
    ur5_2014350238_base.AddFile(PATH + 'STL Files/UR5 Plate.STL')
    ur5_plate = RDK.Item('UR5 Plate', ITEM_TYPE_OBJECT)
    ur5_plate.setVisible(True, False)
    
    table_top = RDK.AddFrame('Table Top', ur5_2014350238_base)
    table_top.setPose(UR_2_Pose([-700, -700, -20, 0, 0, 0]))
    table_top.setVisible(False)
    
    table_top.AddFile(PATH + 'STL Files/Table Plate.STL')
    table_plate = RDK.Item('Table Plate', ITEM_TYPE_OBJECT)
    table_plate.setVisible(True, False)
        
    return

##
#   @brief Add tool stand frames.
#
#   Adds the three tool stand frames used as the basis for tool targets, as
#   well as the two base frames used to locate the tool stand.
#
def add_tool_stand_frames():
    table_top = RDK.Item('Table Top', ITEM_TYPE_FRAME)
    
    tool_stand_plate_base = RDK.AddFrame('Tool Stand Plate Base', table_top)
    tool_stand_plate_base.setPose(UR_2_Pose([100, 800, 0, 0, 0, -0.524]))
    tool_stand_plate_base.setVisible(False)
    
    # The nominal (i.e. SolidWorks) location of the 'Tool Stand Base' frame
    # relative to the 'Tool Stand Base Plate' frame is [63.5, -63.5, ...] but
    # because the true location of the UR5 / UR5 Plate are not yet known, some
    # adjustment is necessary. The [66.6, -64.03, ...] figures used below come
    # from comparing the true location of the 'Grinder Tool Inserted' target
    # to its nominal location. Adjusting the location of the 'Tool Stand Base'
    # frame means there is then no need to adjust the location of the 'Grinder
    # Tool Top' frame.
    tool_stand_base = RDK.AddFrame('Tool Stand Base', tool_stand_plate_base)
    tool_stand_base.setPose(UR_2_Pose([66.6, -64.03, 20, 0, 0, 0]))
    tool_stand_base.setVisible(False)
    
    grinder_tool_top = RDK.AddFrame('Grinder Tool Top', tool_stand_base)
    grinder_tool_top.setPose(UR_2_Pose([-9.25, -100.74, 513, 3.134, 0, 0]))
    grinder_tool_top.setVisible(False)
    
    portafilter_tool_top = RDK.AddFrame('Portafilter Tool Top', tool_stand_base)
    portafilter_tool_top.setPose(UR_2_Pose([-9.25, 64.26, 511.77, 3.134, 0, 0]))
    portafilter_tool_top.setVisible(False)
    
    cup_tool_top = RDK.AddFrame('Cup Tool Top', tool_stand_base)
    cup_tool_top.setPose(UR_2_Pose([-9.25, 229.25, 510.46, 3.134, 0, 0]))
    cup_tool_top.setVisible(False)
    
    return
    
##
#   @brief Add tool stand geometry.
#
#   Adds all tool stand geometry, including the plate to which it is bolted,
#   and the tools that it carries.
#
def add_tool_stand_geometry():
    tool_stand_plate_base = RDK.Item('Tool Stand Plate Base', ITEM_TYPE_FRAME)
    tool_stand_plate_base.AddFile(PATH + 'STL Files/Tool Stand Plate.STL')
    tool_stand_plate = RDK.Item('Tool Stand Plate', ITEM_TYPE_OBJECT)
    tool_stand_plate.setVisible(True, False)

    tool_stand_base = RDK.Item('Tool Stand Base', ITEM_TYPE_FRAME)
    tool_stand_base.AddFile(PATH + 'STL Files/Tool Stand.STL')
    tool_stand = RDK.Item('Tool Stand', ITEM_TYPE_OBJECT)
    tool_stand.setVisible(True, True)
    
    grinder_tool_top = RDK.Item('Grinder Tool Top', ITEM_TYPE_FRAME)
    grinder_tool_top.AddFile(PATH + 'STL Files/Grinder Tool.STL')
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_OBJECT)
    grinder_tool.setVisible(True, False)
    
    portafilter_tool_top = RDK.Item('Portafilter Tool Top', ITEM_TYPE_FRAME)
    portafilter_tool_top.AddFile(PATH + 'STL Files/Portafilter Tool.STL')
    portafilter_tool = RDK.Item('Portafilter Tool', ITEM_TYPE_OBJECT)
    portafilter_tool.setVisible(True, False)
    
    cup_tool_top = RDK.Item('Cup Tool Top', ITEM_TYPE_FRAME)
    cup_tool_top.AddFile(PATH + 'STL Files/Cup Tool Closed.STL')
    cup_tool_closed = RDK.Item('Cup Tool Closed', ITEM_TYPE_OBJECT)
    cup_tool_closed.setVisible(True, False)
    
    return

##
#   @brief Add tool stand targets.
#
#   Adds the targets needed to be able to write programs to attach / detach
#   slave tools to / from the master. Due to the fact that the true location
#   of the tool stand differs to its nominal (i.e. SolidWorks) location, the
#   x-axis and y-axis values of the portafilter and cup tools are offset by
#   a small amount.
#
def add_tool_stand_targets():
    grinder_tool_top = RDK.Item('Grinder Tool Top', ITEM_TYPE_FRAME)
    
    grinder_tool_approach = RDK.AddTarget('Grinder Tool Approach', grinder_tool_top)
    grinder_tool_approach.setPose(UR_2_Pose([150, 0, -20, 0, 0, 0.873]))
    grinder_tool_approach.setAsCartesianTarget()
    grinder_tool_approach.setVisible(False)
    
    grinder_tool_aligned = RDK.AddTarget('Grinder Tool Aligned', grinder_tool_top)
    grinder_tool_aligned.setPose(UR_2_Pose([0, 0, -20, 0, 0, 0.873]))
    grinder_tool_aligned.setAsCartesianTarget()
    grinder_tool_aligned.setVisible(False)
    
    grinder_tool_inserted = RDK.AddTarget('Grinder Tool Inserted', grinder_tool_top)
    grinder_tool_inserted.setPose(UR_2_Pose([0, 0, -2, 0, 0, 0.873]))
    grinder_tool_inserted.setAsCartesianTarget()
    grinder_tool_inserted.setVisible(False)
    
    grinder_tool_mated = RDK.AddTarget('Grinder Tool Mated', grinder_tool_top)
    grinder_tool_mated.setPose(UR_2_Pose([0, 0, 0, 0, 0, 0.873]))
    grinder_tool_mated.setAsCartesianTarget()
    grinder_tool_mated.setVisible(False)
    
    portafilter_tool_top = RDK.Item('Portafilter Tool Top', ITEM_TYPE_FRAME)
    
    portafilter_tool_approach = RDK.AddTarget('Portafilter Tool Approach', portafilter_tool_top)
    portafilter_tool_approach.setPose(UR_2_Pose([150, 0, -20, 0, 0, 0.873]))
    portafilter_tool_approach.setAsCartesianTarget()
    portafilter_tool_approach.setVisible(False)
    
    portafilter_tool_aligned = RDK.AddTarget('Portafilter Tool Aligned', portafilter_tool_top)
    portafilter_tool_aligned.setPose(UR_2_Pose([-0.2, 0.1, -20, 0, 0, 0.873]))
    portafilter_tool_aligned.setAsCartesianTarget()
    portafilter_tool_aligned.setVisible(False)
    
    portafilter_tool_inserted = RDK.AddTarget('Portafilter Tool Inserted', portafilter_tool_top)
    portafilter_tool_inserted.setPose(UR_2_Pose([-0.2, 0.1, -2, 0, 0, 0.873]))
    portafilter_tool_inserted.setAsCartesianTarget()
    portafilter_tool_inserted.setVisible(False)
    
    portafilter_tool_mated = RDK.AddTarget('Portafilter Tool Mated', portafilter_tool_top)
    portafilter_tool_mated.setPose(UR_2_Pose([-0.2, 0.1, 0, 0, 0, 0.873]))
    portafilter_tool_mated.setAsCartesianTarget()
    portafilter_tool_mated.setVisible(False)
    
    cup_tool_top = RDK.Item('Cup Tool Top', ITEM_TYPE_FRAME)
    
    cup_tool_approach = RDK.AddTarget('Cup Tool Approach', cup_tool_top)
    cup_tool_approach.setPose(UR_2_Pose([150, 0, -20, 0, 0, 0.873]))
    cup_tool_approach.setAsCartesianTarget()
    cup_tool_approach.setVisible(False)
    
    cup_tool_aligned = RDK.AddTarget('Cup Tool Aligned', cup_tool_top)
    cup_tool_aligned.setPose(UR_2_Pose([0.5, 0.4, -20, 0, 0, 0.873]))
    cup_tool_aligned.setAsCartesianTarget()
    cup_tool_aligned.setVisible(False)
    
    cup_tool_inserted = RDK.AddTarget('Cup Tool Inserted', cup_tool_top)
    cup_tool_inserted.setPose(UR_2_Pose([0.5, 0.4, -2, 0, 0, 0.873]))
    cup_tool_inserted.setAsCartesianTarget()
    cup_tool_inserted.setVisible(False)
    
    cup_tool_mated = RDK.AddTarget('Cup Tool Mated', cup_tool_top)
    cup_tool_mated.setPose(UR_2_Pose([0.5, 0.4, 0, 0, 0, 0.873]))
    cup_tool_mated.setAsCartesianTarget()
    cup_tool_mated.setVisible(False)
    
    cup_tool_attached = RDK.AddTarget('Cup Tool Attached', cup_tool_top)
    cup_tool_attached.setPose(UR_2_Pose([0.5, 0.4, -13, 0, 0, 0.873]))
    cup_tool_attached.setAsCartesianTarget()
    cup_tool_attached.setVisible(False)
    
    cup_tool_rotated = RDK.AddTarget('Cup Tool Rotated', cup_tool_top)
    cup_tool_rotated.setPose(UR_2_Pose([0.5, 0.4, -13, 0, 0, 2.443]))
    cup_tool_rotated.setAsCartesianTarget()
    cup_tool_rotated.setVisible(False)
    
    cup_tool_depart = RDK.AddTarget('Cup Tool Depart', cup_tool_top)
    cup_tool_depart.setPose(UR_2_Pose([150.5, 0.4, -13, 0, 0, 2.443]))
    cup_tool_depart.setAsCartesianTarget()
    cup_tool_depart.setVisible(False)
    
    return

##
#   Add grinder attach program to the station.
#
#   In order to avoid damaging the tooling, students must use the program
#   generated by this function to attach the grinder tool to the robot.
#
def add_grinder_attach_program():
    grinder_tool_attach = RDK.AddProgram('Grinder Tool Attach')    
    
    grinder_tool_attach.setPoseFrame(RDK.Item('Grinder Tool Top', ITEM_TYPE_FRAME))
    grinder_tool_attach.setPoseTool(RDK.Item('Master Tool', ITEM_TYPE_TOOL))
    
    grinder_tool_attach.MoveJ(RDK.Item('Grinder Tool Approach', ITEM_TYPE_TARGET))
    grinder_tool_attach.MoveL(RDK.Item('Grinder Tool Aligned', ITEM_TYPE_TARGET))
    grinder_tool_attach.MoveL(RDK.Item('Grinder Tool Inserted', ITEM_TYPE_TARGET))
    
    grinder_tool_attach.setDO(ATTACH_TOOL, 1)
    grinder_tool_attach.Pause(500)
    
    grinder_tool_attach.MoveL(RDK.Item('Grinder Tool Mated', ITEM_TYPE_TARGET))
    
    # This is the code that does not work \/
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_TOOL)
    grinder_tool.setVisible(True, True)
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_OBJECT)
    grinder_tool.setVisible(False)
    # This is the code that does not work /\
    
    grinder_tool_attach.MoveL(RDK.Item('Grinder Tool Aligned', ITEM_TYPE_TARGET))
    grinder_tool_attach.MoveJ(RDK.Item('Grinder Tool Approach', ITEM_TYPE_TARGET))
    
    grinder_tool_attach.setPoseTool(RDK.Item('Grinder Tool', ITEM_TYPE_TOOL))
    grinder_tool_attach.setPoseFrame(RDK.Item('UR5 2014350238 Base', ITEM_TYPE_FRAME))
    
    return

##
#   Add grinder detach program to the station.
#
#   In order to avoid damaging the tooling, students must use the program
#   generated by this function to detach the grinder tool from the robot.
#
def add_grinder_detach_program():
    grinder_tool_detach = RDK.AddProgram('Grinder Tool Detach')
    
    grinder_tool_detach.setPoseFrame(RDK.Item('Grinder Tool Top', ITEM_TYPE_FRAME))
    grinder_tool_detach.setPoseTool(RDK.Item('Master Tool', ITEM_TYPE_TOOL))
    
    grinder_tool_detach.MoveJ(RDK.Item('Grinder Tool Approach', ITEM_TYPE_TARGET))
    grinder_tool_detach.MoveL(RDK.Item('Grinder Tool Aligned', ITEM_TYPE_TARGET))
    grinder_tool_detach.MoveL(RDK.Item('Grinder Tool Mated', ITEM_TYPE_TARGET))
    
    grinder_tool_detach.setDO(ATTACH_TOOL, 0)
    grinder_tool_detach.Pause(500)
    grinder_tool_detach.setDO(DETACH_TOOL, 1)
    grinder_tool_detach.Pause(500)
    grinder_tool_detach.setDO(DETACH_TOOL, 0)
    
    # This is the code that does not work \/
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_TOOL)
    grinder_tool.setVisible(False)
    grinder_tool = RDK.Item('Grinder Tool', ITEM_TYPE_OBJECT)
    grinder_tool.setVisible(True, False)
    # This is the code that does not work /\
    
    grinder_tool_detach.MoveL(RDK.Item('Grinder Tool Aligned', ITEM_TYPE_TARGET))
    grinder_tool_detach.MoveJ(RDK.Item('Grinder Tool Approach', ITEM_TYPE_TARGET))
    
    grinder_tool_detach.setPoseFrame(RDK.Item('UR5 2014350238 Base', ITEM_TYPE_FRAME))
    
    return
    
#-----------------------------------------------------------------------------
#   Program
#-----------------------------------------------------------------------------
RDK = Robolink()

add_ur5_robot()

add_ur5_master_tool()
add_ur5_slave_tools()
add_ur5_paper_cup()

set_master_tool_pose()

add_table()
add_tool_stand_frames()
add_tool_stand_geometry()
add_tool_stand_targets()

add_grinder_attach_program()
add_grinder_detach_program()
